/**
  ******************************************************************************
  * @file    main.c
  * @author  MCU Application Team
  * @brief   Main program body
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2023 Puya Semiconductor Co.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by Puya under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2016 STMicroelectronics.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by ST under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  */

/* Includes ------------------------------------------------------------------*/
#include "main.h"

/* Private define ------------------------------------------------------------*/
#define TRANSMITTER_BOARD

/* Private variables ---------------------------------------------------------*/
LPUART_HandleTypeDef hlpuart;
DMA_HandleTypeDef hdma_lpuart_rx;
DMA_HandleTypeDef hdma_lpuart_tx;

/* Buffer used for transmission */
uint8_t aTxBuffer[] = " ****LPUART_TwoBoards_ComDMA**** ";
/* Buffer used for reception */
uint8_t aRxBuffer[RXBUFFERSIZE];

/* Private user code ---------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
static void APP_SystemClockConfig(void);
static void APP_LPUART_Init(void);
static uint16_t Buffercmp(uint8_t *pBuffer1, uint8_t *pBuffer2, uint16_t BufferLength);

/**
  * @brief  Main program.
  * @retval int
  */
int main(void)
{
  /* Reset of all peripherals, Initializes the Systick. */ 
  HAL_Init();

  /* Clock init */
  APP_SystemClockConfig();

  /* Initialize LED */
  BSP_LED_Init(LED_GREEN);

  /* Initialize LPUART */
  APP_LPUART_Init();

#ifdef TRANSMITTER_BOARD
  /* Initialization button */
  BSP_PB_Init(BUTTON_KEY,BUTTON_MODE_GPIO);

  /* Wait for the button to be pressed */
  while(BSP_PB_GetState(BUTTON_KEY) == 0);

  /* The board sends the message and expects to receive it back */
  /* Start the transmission process */
  if(HAL_LPUART_Transmit_DMA(&hlpuart, (uint8_t*)aTxBuffer, TXBUFFERSIZE)!= HAL_OK)
  {
    APP_ErrorHandler();
  }

  /* Wait for the end of the transfer */
  while(HAL_LPUART_GetState(&hlpuart) != HAL_LPUART_STATE_READY)
  {
  }

  /* Put LPUART peripheral in reception process */
  if(HAL_LPUART_Receive_DMA(&hlpuart, (uint8_t *)aRxBuffer, RXBUFFERSIZE) != HAL_OK)
  {
    APP_ErrorHandler();
  }
#else
  /* The board receives the message and sends it back */
  /* Put LPUART peripheral in reception process */
  if(HAL_LPUART_Receive_DMA(&hlpuart, (uint8_t *)aRxBuffer, RXBUFFERSIZE) != HAL_OK)
  {
    APP_ErrorHandler();
  }

  /* Wait for the end of the transfer */
  while(HAL_LPUART_GetState(&hlpuart) != HAL_LPUART_STATE_READY)
  {
  }

  /* Wait some ms for synchronization */
  HAL_Delay(100);

  /* Start the transmission process */
  if(HAL_LPUART_Transmit_DMA(&hlpuart, (uint8_t*)aTxBuffer, TXBUFFERSIZE)!= HAL_OK)
  {
    APP_ErrorHandler();
  }

#endif /* TRANSMITTER_BOARD */

  /* Wait for the end of the transfer */
  while(HAL_LPUART_GetState(&hlpuart) != HAL_LPUART_STATE_READY)
  {
  }

  /* Compare the sent and received buffers */
  if(Buffercmp((uint8_t*)aTxBuffer,(uint8_t*)aRxBuffer,RXBUFFERSIZE))
  {
    APP_ErrorHandler();
  }

  /* Turn LED on: Transfer process is correct */
  BSP_LED_On(LED_GREEN);

  while (1)
  {
  }  
}

/**
  * @brief LPUART1 Initialization Function
  * @param None
  * @retval None
  */
static void APP_LPUART_Init(void)
{
  hlpuart.Instance = LPUART1;
  hlpuart.Init.BaudRate = 9600;
  hlpuart.Init.WordLength = LPUART_WORDLENGTH_8B;
  hlpuart.Init.StopBits = LPUART_STOPBITS_1;
  hlpuart.Init.Parity = LPUART_PARITY_NONE;
  hlpuart.Init.Mode = LPUART_MODE_TX_RX;
  hlpuart.Init.HwFlowCtl = LPUART_HWCONTROL_NONE;
  hlpuart.AdvancedInit.AdvFeatureInit = LPUART_ADVFEATURE_NO_INIT;
  if (HAL_LPUART_Init(&hlpuart) != HAL_OK)
  {
    APP_ErrorHandler();
  }
}

/**
  * @brief  Compares two buffers.
  * @param  pBuffer1, pBuffer2: buffers to be compared.
  * @param  BufferLength: buffer's length
  * @retval 0  : pBuffer1 identical to pBuffer2
  *         >0 : pBuffer1 differs from pBuffer2
  */
static uint16_t Buffercmp(uint8_t *pBuffer1, uint8_t *pBuffer2, uint16_t BufferLength)
{
  while (BufferLength--)
  {
    if ((*pBuffer1) != *pBuffer2)
    {
      return BufferLength;
    }
    pBuffer1++;
    pBuffer2++;
  }

  return 0;
}

/**
  * @brief  System clock configuration function
  * @param  None
  * @retval None
  */
static void APP_SystemClockConfig(void)
{
  RCC_OscInitTypeDef RCC_OscInitStruct = {0};
  RCC_ClkInitTypeDef RCC_ClkInitStruct = {0};

  /* Oscillator configuration */
  RCC_OscInitStruct.OscillatorType = RCC_OSCILLATORTYPE_HSI | RCC_OSCILLATORTYPE_LSI | RCC_OSCILLATORTYPE_MSI; /* Select oscillator HSI, LSI, MSI */
#if defined(RCC_HSE_SUPPORT)
  RCC_OscInitStruct.OscillatorType |= RCC_OSCILLATORTYPE_HSE; /* Select oscillator HSE */
#endif /* RCC_HSE_SUPPORT */
#if defined(RCC_LSE_SUPPORT)
  RCC_OscInitStruct.OscillatorType |= RCC_OSCILLATORTYPE_LSE; /* Select oscillator LSE */
#endif /* RCC_LSE_SUPPORT */
  RCC_OscInitStruct.HSIState = RCC_HSI_ON;                          /* Enable HSI */
  RCC_OscInitStruct.HSIDiv = RCC_HSI_DIV1;                          /* HSI 1 frequency division */
  RCC_OscInitStruct.HSICalibrationValue = RCC_HSICALIBRATION_8MHz;  /* Configure HSI clock 8MHz */
#if defined(RCC_HSE_SUPPORT)
  RCC_OscInitStruct.HSEState = RCC_HSE_OFF;                         /* Close HSE */
  /*RCC_OscInitStruct.HSEFreq = RCC_HSE_24_32MHz;*/
#endif /* RCC_HSE_SUPPORT */
  RCC_OscInitStruct.LSIState = RCC_LSI_OFF;                         /* Close LSI */
#if defined(RCC_LSE_SUPPORT)
  RCC_OscInitStruct.LSEState = RCC_LSE_OFF;                         /* Close LSE */
  /*RCC_OscInitStruct.LSEDriver = RCC_LSEDRIVE_MEDIUM;*/
#endif /* RCC_LSE_SUPPORT */
  RCC_OscInitStruct.MSIState = RCC_MSI_OFF;                         /* Close MSI */
  /*RCC_OscInitStruct.MSIDiv = RCC_MSI_DIV1;*/
  RCC_OscInitStruct.PLL.PLLState = RCC_PLL_OFF;                     /* Close PLL */
  /*RCC_OscInitStruct.PLL.PLLSource = RCC_PLLSOURCE_HSI;*/
  /*RCC_OscInitStruct.PLL.PLLMUL = RCC_PLL_MUL2;*/
  /* Configure oscillator */
  if (HAL_RCC_OscConfig(&RCC_OscInitStruct) != HAL_OK)
  {
    APP_ErrorHandler();
  }

  /* Clock source configuration */
  RCC_ClkInitStruct.ClockType = RCC_CLOCKTYPE_HCLK | RCC_CLOCKTYPE_SYSCLK | RCC_CLOCKTYPE_PCLK; /* Choose to configure clock HCLK, SYSCLK, PCLK */
  RCC_ClkInitStruct.SYSCLKSource = RCC_SYSCLKSOURCE_HSISYS; /* Select HSISYS as the system clock */
  RCC_ClkInitStruct.AHBCLKDivider = RCC_SYSCLK_DIV1;        /* AHB clock 1 division */
  RCC_ClkInitStruct.APBCLKDivider = RCC_HCLK_DIV1;          /* APB clock 1 division */
  /* Configure clock source */
  if (HAL_RCC_ClockConfig(&RCC_ClkInitStruct, FLASH_LATENCY_0) != HAL_OK)
  {
    APP_ErrorHandler();
  }
}

/**
  * @brief  Error executing function.
  * @param  None
  * @retval None
  */
void APP_ErrorHandler(void)
{
  while (1)
  {
  }
}

#ifdef  USE_FULL_ASSERT
/**
  * @brief  Reports the name of the source file and the source line number
  *         where the assert_param error has occurred.
  * @param  file: pointer to the source file name
  * @param  line: assert_param error line source number
  * @retval None
  */
void assert_failed(uint8_t *file, uint32_t line)
{
  /* Users can add their own printing information as needed,
     for example: printf("Wrong parameters value: file %s on line %d\r\n", file, line) */
  /* Infinite loop */
  while (1)
  {
  }
}
#endif /* USE_FULL_ASSERT */

/************************ (C) COPYRIGHT Puya *****END OF FILE******************/
